#include <cstdint>

#pragma once

typedef bool		PMC_BOOL;
typedef uint8_t		PMC_BYTE;
typedef uint16_t	PMC_WORD;
typedef uint32_t	PMC_DWORD;
typedef uint64_t	PMC_LWORD;
typedef uint8_t		PMC_USINT;
typedef uint16_t	PMC_UINT;
typedef uint32_t	PMC_UDINT;
typedef uint64_t	PMC_ULINT;
typedef int8_t		PMC_SINT;
typedef int16_t		PMC_INT;
typedef int32_t		PMC_DINT;
typedef int64_t		PMC_LINT;
typedef float		PMC_REAL;
typedef double		PMC_LREAL;

namespace PMC
{ 
	/// <summary>
	/// Enum for the source of the trigger for WaitUntil command
	/// </summary>
	enum class TRIGGERSOURCE : PMC_USINT
	{
		/// <summary>
		/// Time delay trigger mode
		/// </summary>
		TIME_DELAY,
		/// <summary>
		/// external digital input trigger mode
		/// </summary>
		EXTERNAL_DI,
		/// <summary>
		/// fieldbus digital input trigger mode
		/// </summary>
		FIELDBUS_DI,
		/// <summary>
		/// command label trigger mode
		/// </summary>
		CMD_LABEL,
		/// <summary>
		/// displacement trigger mode
		/// </summary>
		DISPLACEMENT
	};

	/// <summary>
	/// Enum for when the WaitUntil command should start monitoring
	/// </summary>
	enum class WAITUNTILSTARTOPTION : PMC_USINT
	{
		/// <summary>
		/// Starts monitoring when the command is executed (removed from the buffer)
		/// </summary>
		STARTATEXECUTION,
		/// <summary>
		/// Starts monitoring when the command is received (added in the buffer)
		/// </summary>
		STARTATRECEIVED,
		/// <summary>
		/// Starts monitoring for the finish condition only after the specified start condition has been met
		/// </summary>
		SPECIFYSTARTCONDITION
	};

	/// <summary>
	/// enum for possible trigger edge types for digital inputs
	/// </summary>
	enum class TRIGGEREDGETYPE : PMC_USINT
	{
		/// <summary>
		/// triggers on the rising edge
		/// </summary>
		RISING_EDGE,
		/// <summary>
		/// triggers on the falling edge
		/// </summary>
		FALLING_EDGE,
		/// <summary>
		/// triggers when source value is 1
		/// </summary>
		IS_ONE,
		/// <summary>
		/// triggers when source value is 0
		/// </summary>
		IS_ZERO
	};

	/// <summary>
	/// enum for when to trigger during command label delay method
	/// </summary>
	enum class TRIGGERCMDLABELTYPE : PMC_USINT
	{
		/// <summary>
		/// triggers at the start of command
		/// </summary>
		CMD_START,
		/// <summary>
		/// triggers when the command has finished
		/// </summary>
		CMD_FINISH,
		/// <summary>
		/// triggers during command execution
		/// </summary>
		CMD_EXECUTING
	};

	/// <summary>
	/// enum for defining trigger command label's command type
	/// </summary>
	enum class TRIGGERCMDTYPE : PMC_USINT
	{
		/// <summary>
		/// The command is a standard motion command such as linear motion, arc motion, etc.
		/// </summary>
		MOTION_COMMAND,
		/// <summary>
		/// The command is a run macro motion command
		/// </summary>
		RUNMACRO_COMMAND
	};

	/// <summary>
	/// enum for trigger type during displacement trigger method
	/// </summary>
	enum class TRIGGERDISPLACEMENTTYPE : PMC_USINT
	{
		/// <summary>
		/// triggers when displacement (ax+by, or only x, or only y) is greater than threshold
		/// </summary>
		GREATER_THAN,
		/// <summary>
		/// triggers when displacement (ax+by, or only x, or only y) is less than threshold
		/// </summary>
		LESS_THAN,
		/// <summary>
		/// triggers when going from less than threshold to greater than threshold
		/// </summary>
		POSITIVE_CROSS,
		/// <summary>
		/// triggers when going from greater than threshold to less than threshold
		/// </summary>
		NEGATIVE_CROSS
	};

	/// <summary>
	/// enum for trigger mode during displacement trigger method
	/// </summary>
	enum class TRIGGERDISPLACEMENTMODE : PMC_USINT
	{
		/// <summary>
		/// monitoring X position only
		/// </summary>
		X_ONLY,
		/// <summary>
		/// monitoring Y position only
		/// </summary>
		Y_ONLY,
		/// <summary>
		/// monitoring by an arbitrary line
		/// </summary>
		AX_BY
	};

	/// <summary>
	/// can choose between position and force feedback in the GetXBOTStatus command
	/// </summary>
	enum class FEEDBACKOPTION : PMC_USINT
	{
		/// <summary>
		/// provides position data in the command reply
		/// </summary>
		POSITION,
		/// <summary>
		/// provides force data in the command reply
		/// </summary>
		FORCE
	};

	/// <summary>
	/// reply from the PMC for a given command
	/// </summary>
	enum class PMCRTN : PMC_UINT
	{
		/// <summary>
		/// command accepted
		/// </summary>
		ALLOK = 0,
		/// <summary>
		/// PMC failed to reply, please resend command
		/// </summary>
		SYSTEMERROR = 0X0001,
		/// <summary>
		/// PMC is in an incorrect state for this command. (For example, not activated)
		/// </summary>
		WRONGPMCSTATE = 0x2000,
		/// <summary>
		/// Do not have the required mastership for this command
		/// </summary>
		NOMASTERSHIP = 0x2001,
		/// <summary>
		/// mastership did not provide a response
		/// </summary>
		MASTERSHIP_TIMEOUT = 0x2002,
		/// <summary>
		/// Group command rejected due to wrong group state
		/// </summary>
		WRONG_GROUP_STATE = 0x2003,
		/// <summary>
		/// Macro command rejected due to wrong macro state
		/// </summary>
		WRONG_MACRO_STATE = 0x2004,
		/// <summary>
		/// Wrong digital input / output state
		/// </summary>
		WRONG_DIGITAL_IO_STATE = 0x2005,
		/// <summary>
		/// wrong flyway state
		/// </summary>
		WRONG_FLYWAY_STATE = 0x2006,
		/// <summary>
		/// Failed to find a routing solution for auto driving
		/// </summary>
		NO_ROUTING_SOLUTION = 0x2008,
		/// <summary>
		/// Failed to find a routing solution for auto driving
		/// </summary>
		COMMUNICATION_TIMEOUT = 0x2009,
		/// <summary>
		/// missing license
		/// </summary>
		NO_LICENSE = 0x200A,
		/// <summary>
		/// mover stereotype is not defined
		/// </summary>
		UNDEFINED_STEREOTYPE = 0x200B,
		/// <summary>
		/// Xbot state incorrect for this command. (For example, not levitated)
		/// </summary>
		WRONGXBOTSTATE = 0x3000,
		/// <summary>
		/// Parameters for this command are invalid
		/// </summary>
		INVALIDPARAMS = 0x4000,
		/// <summary>
		/// wrong queue state
		/// </summary>
		WRONG_QUEUE_STATE = 0x5000,
		/// <summary>
		/// The zone is in the wrong state to perform this operation
		/// </summary>
		WRONG_ZONE_STATE = 0x5001,
		/// <summary>
		/// An XBot is located on the zone boundary
		/// </summary>
		ZONE_NOT_AVAILABLE = 0x5002,
		/// <summary>
		/// only used for code generation, not an actual command for the PMC
		/// </summary>
		CODE_GENERATION_ONLY = 0xFFFE,
		/// <summary>
		/// the command ID is invalid
		/// </summary>
		INVALIDCOMMAND = 0xFFFF
	};

	/// <summary>
	/// XBot state enum
	/// </summary>
	enum class XBOTSTATE : PMC_USINT
	{
		/// <summary>
		/// XBot not detected by the PMC
		/// </summary>
		XBOT_UNDETECTED = 0,
		/// <summary>
		/// XBOT is landed
		/// </summary>
		XBOT_DISCOVERING,
		/// <summary>
		/// XBOT is landed
		/// </summary>
		XBOT_LANDED,
		/// <summary>
		/// XBOT is idling (levitated)
		/// </summary>
		XBOT_IDLE,
		/// <summary>
		/// XBOT is disabled
		/// </summary>
		XBOT_DISABLED,
		/// <summary>
		/// XBOT is in motion
		/// </summary>
		XBOT_MOTION,
		/// <summary>
		/// XBOT is waiting for a trigger
		/// </summary>
		XBOT_WAIT,
		/// <summary>
		/// XBOT is attempting to stop
		/// </summary>
		XBOT_STOPPING,
		/// <summary>
		/// XBOT detected an obstacle and is waiting for obstacle to clear
		/// </summary>
		XBOT_OBSTACLE_DETECTED,
		/// <summary>
		/// XBOT is hold position due excessive error during motion
		/// </summary>
		XBOT_HOLDPOSITION,
		/// <summary>
		/// XBOT stopped and will not automatically resume
		/// </summary>
		XBOT_STOPPED,
		/// <summary>
		/// XBOT is in stream motion
		/// </summary>
		XBOT_RESERVED,
		/// <summary>
		/// XBOT is in asynchronous motion (automatically driven to target)
		/// </summary>
		XBOT_RESERVED1,
		/// <summary>
		/// reserved
		/// </summary>
		XBOT_RESERVED2,
		/// <summary>
		/// XBOT has an error
		/// </summary>
		XBOT_ERROR,
		/// <summary>
		/// XBOT is not installed
		/// </summary>
		XBOT_UNINSTALLED
	};

	/// <summary>
	/// PMC Status Enum
	/// </summary>
	enum class PMCSTATUS : PMC_USINT
	{
		/// <summary>
		/// PMC is booting up
		/// </summary>
		PMC_BOOTING = 0,
		/// <summary>
		/// PMC is inactive (XBOTs deactivated)
		/// </summary>
		PMC_INACTIVE,
		/// <summary>
		/// PMC is starting to activate
		/// </summary>
		PMC_ACTIVATING,
		/// <summary>
		/// Reserved
		/// </summary>
		PMC_RESERVED,
		/// <summary>
		/// Reserved
		/// </summary>
		PMC_SERVICE,
		/// <summary>
		/// PMC in fully controlled mode
		/// </summary>
		PMC_FULLCTRL,
		/// <summary>
		/// PMC in intelligent control mode
		/// </summary>
		PMC_INTELLIGENTCTRL,
		/// <summary>
		/// PMC is deactivating
		/// </summary>
		PMC_DEACTIVATING,
		/// <summary>
		/// PMC is handling an error and recording snapshots
		/// </summary>
		PMC_ERRORHANDLING,
		/// <summary>
		/// PMC is in the error state
		/// </summary>
		PMC_ERROR,
	};

	/// <summary>
	/// Multi-PMC Border Status Enum
	/// </summary>
	enum class BORDERSTATUS : PMC_USINT
	{
		/// <summary>
		/// The flyway link between this border is not connected
		/// </summary>
		DISCONNECTED = 0,
		/// <summary>
		/// The flyway link between this border is connected, but it is not yet ready to receive xbots
		/// </summary>
		NOTREADY,
		/// <summary>
		/// The flyway link at this border is connected and it is ready to receive xbots
		/// </summary>
		READY
	};

	/// <summary>
	/// Motion position mode Enum
	/// </summary>
	enum class POSITIONMODE : PMC_USINT
	{
		/// <summary>
		/// Absolute positioning
		/// </summary>
		ABSOLUTE_MOTION = 0,
		/// <summary>
		/// Relative positioning
		/// </summary>
		RELATIVE_MOTION
	};

	/// <summary>
	/// Can choose between
	/// </summary>
	enum class CAMMODE : PMC_USINT
	{
		/// <summary>
		/// automatically trigger the cam motion when the master xbot is inside the cam region
		/// </summary>
		AutoStart = 0,
		/// <summary>
		/// the profile is unwound and repeats cyclicly, such that the cam profile covers the range from -infinity to + infinity for the master axis
		/// </summary>
		Cyclic = 1,
		/// <summary>
		/// the cam stops automatically if the master axis exits the defined cam region, and does not start again if the master axis re-enters the cam region
		/// </summary>
		SingleStart = 2,
	};

	/// <summary>
	/// Select the cam engagement direction
	/// </summary>
	enum class CAMRATCHETDIRECTION : PMC_USINT
	{
		/// <summary>
		/// The cam is engaged whether the master axis value is changing in the forward (positive) or backward (negative) direction
		/// </summary>
		BOTH_DIRECTIONS = 0,
		//future editions will add forward only and reverse only

	};

	/// <summary>
	/// Linear path type Enum
	/// </summary>
	enum class LINEARPATHTYPE : PMC_USINT
	{
		/// <summary>
		/// Direct motion
		/// </summary>
		DIRECT = 0,
		/// <summary>
		/// X motion first, then Y motion
		/// </summary>
		XTHENY,
		/// <summary>
		/// Y motion first, then X motion
		/// </summary>
		YTHENX
	};

	/// <summary>
	/// Arc direction Enum
	/// </summary>
	enum class ARCDIRECTION : PMC_USINT
	{
		/// <summary>
		/// Clockwise motion
		/// </summary>
		CLOCKWISE = 0,
		/// <summary>
		/// Counter-clockwise motion
		/// </summary>
		COUNTERCLOCKWISE
	};

	/// <summary>
	/// Arc type Enum
	/// </summary>
	enum class ARCTYPE : PMC_USINT
	{
		/// <summary>
		/// Minor arc
		/// </summary>
		MINORARC = 0,
		/// <summary>
		/// Major arc
		/// </summary>
		MAJORARC
	};

	/// <summary>
	/// Arc motion mode Enum
	/// </summary>
	enum class ARCMODE : PMC_USINT
	{
		/// <summary>
		/// Target + radius mode
		/// </summary>
		TARGETRADIUS = 0,
		/// <summary>
		/// Center + angle mode
		/// </summary>
		CENTERANGLE
	};

	/// <summary>
	/// Motion buffer options Enum
	/// </summary>
	enum class MOTIONBUFFEROPTIONS : PMC_USINT
	{
		/// <summary>
		/// Block motion buffer
		/// </summary>
		BLOCKBUFFER = 0,
		/// <summary>
		/// Release motion buffer
		/// </summary>
		RELEASEBUFFER,
		/// <summary>
		/// Clear motion buffer
		/// </summary>
		CLEARBUFFER
	};

	/// <summary>
	/// Motion macro options Enum
	/// </summary>
	enum class MOTIONMACROOPTIONS : PMC_USINT
	{
		/// <summary>
		/// Clear motion macro
		/// </summary>
		CLEARMACRO = 0,
		/// <summary>
		/// Finish motion macro editing
		/// </summary>
		SAVEMACRO = 2,
		/// <summary>
		/// Query status of macro
		/// </summary>
		QUERYSTATUS = 4
	};

	/// <summary>
	/// Group options Enum
	/// </summary>
	enum class GROUPOPTIONS : PMC_USINT
	{
		/// <summary>
		/// Create group
		/// </summary>
		CREATEGROUP = 0,
		/// <summary>
		/// Delete group
		/// </summary>
		DELETEGROUP,
		/// <summary>
		/// Connect group
		/// </summary>
		BONDGROUP,
		/// <summary>
		/// Disconnect group
		/// </summary>
		UNBONDGROUP,
		/// <summary>
		/// Block motion buffer for group
		/// </summary>
		BLOCKMEMBERSBUFFER,
		/// <summary>
		/// Unblock motion buffer for group
		/// </summary>
		RELEASEMEMBERSBUFFER,
		/// <summary>
		/// Query group status
		/// </summary>
		QUERYSTATUS
	};

	/// <summary>
	/// Levitation options Enum
	/// </summary>
	enum class LEVITATEOPTIONS : PMC_USINT
	{
		/// <summary>
		/// Land XBOT
		/// </summary>
		LAND = 0,
		/// <summary>
		/// Levitate XBOT
		/// </summary>
		LEVITATE
	};

	/// <summary>
	/// Levitation speed options enum
	/// </summary>
	enum class LEVITATIONSPEED : PMC_USINT
	{
		/// <summary>
		/// Levitate/Land the xbot(s) in roughly 1.6s
		/// </summary>
		APPROX_1600MS = 0,
		/// <summary>
		/// Levitate/Land the xbot(s) in roughly 0.8s
		/// </summary>
		APPROX_800MS,
		/// <summary>
		/// Levitate/Land the xbot(s) in roughly 0.4s
		/// </summary>
		APPROX_400MS,
		/// <summary>
		/// Levitate/Land the xbot(s) in roughly 0.2s
		/// </summary>
		APPROX_200MS,
		/// <summary>
		/// Levitate/Land the xbot(s) in roughly 0.1s
		/// </summary>
		APPROX_100MS,
		/// <summary>
		/// Levitate/Land the xbot(s) in roughly 0.05s
		/// </summary>
		APPROX_50MS
	};

	/// <summary>
	/// Mobility options enums
	/// </summary>
	enum class MOBILITYOPTIONS : PMC_USINT
	{
		/// <summary>
		/// Disable XBOT
		/// </summary>
		DISABLE = 0,
		/// <summary>
		/// Land XBOT
		/// </summary>
		LAND = 1,
		/// <summary>
		/// Levitate XBOT
		/// </summary>
		LEVITATE = 2
	};

	/// <summary>
	/// motion interrupt options Enum
	/// </summary>
	enum class MOTIONINTERRUPTOPTIONS : PMC_USINT
	{
		/// <summary>
		/// Resume the xbot motion
		/// </summary>
		RESUME = 0,
		/// <summary>
		/// pause the xbot motion
		/// </summary>
		PAUSE
	};

	/// <summary>
	/// Async motion options Enum
	/// </summary>
	enum class ASYNCOPTIONS : PMC_USINT
	{
		/// <summary>
		/// All XBOTs on flyways can be moved
		/// </summary>
		MOVEALL = 0
	};

	/// <summary>
	/// Planet Motion Enum
	/// </summary>
	enum class PLANETOPTIONS : PMC_USINT
	{
		/// <summary>
		/// Remove the planet XBOTs listed in this command from the sun XBOT's planet list
		/// </summary>
		REMOVEPLANETS = 0,
		/// <summary>
		/// add the planet XBOTs listed in this command to the sun XBOT's planet list
		/// </summary>
		ADDPLANETS
	};

	/// <summary>
	/// Mover type enumeration
	/// </summary>
	enum class MOVERTYPE : PMC_USINT
	{
		/// <summary>
		/// M3-06 120mm x 120mm
		/// </summary>
		M3_06 = 0,
		/// <summary>
		/// Reserved
		/// </summary>
		RESERVED1,
		/// <summary>
		/// M3-08 180mm x 120mm
		/// </summary>
		M3_08,
		/// <summary>
		/// M3-08 120mm x 180mm
		/// </summary>
		RESERVED3,
		/// <summary>
		/// Reserved
		/// </summary>
		M3_09X,
		/// <summary>
		/// Reserved
		/// </summary>
		M3_09Y,
		/// <summary>
		/// M3-10 180mm x 180mm
		/// </summary>
		M3_10,
		/// <summary>
		/// Reserved
		/// </summary>
		RESERVED7,
		/// <summary>
		/// M3-11 210mm x 180mm
		/// </summary>
		M3_11X,
		/// <summary>
		/// M3-11 180mm x 210mm
		/// </summary>
		M3_11Y,
		/// <summary>
		/// Reserved
		/// </summary>
		RESERVED10,
		/// <summary>
		/// Reserved
		/// </summary>
		RESERVED11,
		/// <summary>
		/// M3-12 210mm x 210mm
		/// </summary>
		M3_12,
		/// <summary>
		/// Reserved
		/// </summary>
		RESERVED13,
		/// <summary>
		/// M3-13 240mm x 240mm
		/// </summary>
		M3_13,
		/// <summary>
		/// Reserved
		/// </summary>
		RESERVED15,
		/// <summary>
		/// M3-17 300mm x 300mm
		/// </summary>
		M3_17,
		/// <summary>
		/// Reserved
		/// </summary>
		RESERVED17,
		/// <summary>
		/// M3-18 330mm x 330mm
		/// </summary>
		M3_18,
		/// <summary>
		/// Reserved
		/// </summary>
		RESERVED19,
		/// <summary>
		/// Reserved
		/// </summary>
		RESERVED20,
		/// <summary>
		/// Reserved
		/// </summary>
		RESERVED21
	};

	/// <summary>
	/// Enum containing the index of mover properties
	/// </summary>
	enum class MOVERPROPERTY : PMC_USINT
	{
		/// <summary>
		/// Mover type 
		/// </summary>
		MOVERTYPE_0 = 0,
		/// <summary>
		/// Configure payload on the mover, in kg
		/// </summary>
		PAYLOADKG_1,
		/// <summary>
		/// the height of the center of gravity of the payload, in meters
		/// </summary>
		CGHEIGHTM_2,
		/// <summary>
		/// The max size of the payload or the mover, in the X direction, whichever is larger
		/// </summary>
		XDIMENSIONM_3,
		/// <summary>
		/// Reserved for future use
		/// </summary>
		RESERVED_4,
		/// <summary>
		/// The max size of the payload or the mover, in the Y direction, whichever is larger
		/// </summary>
		YDIMENSIONM_5 = 5,
		/// <summary>
		/// Reserved for future use
		/// </summary>
		RESERVED2_6,
		/// <summary>
		/// The maximum acceleration achievable by the mover, with the currently configured payload
		/// </summary>
		MAXACCELERATION_7 = 7
	};

	/// <summary>
	/// Axis enum, base 1
	/// </summary>
	enum class AXISNAMES : PMC_USINT
	{
		/// <summary>
		/// No axis selected
		/// </summary>
		NoAxis_0,
		/// <summary>
		/// X Axis = 1
		/// </summary>
		X_1 = 1,
		/// <summary>
		/// Y Axis = 2
		/// </summary>
		Y_2,
		/// <summary>
		/// Z Axis = 3
		/// </summary>
		Z_3,
		/// <summary>
		/// RX Axis = 4
		/// </summary>
		RX_4,
		/// <summary>
		/// RY Axis = 5
		/// </summary>
		RY_5,
		/// <summary>
		/// RZ Axis = 6
		/// </summary>
		RZ_6
	};

	/// <summary>
	/// cam motion enum
	/// </summary>
	enum class CAMOPTIONS : PMC_USINT
	{
		/// <summary>
		/// Stop cam motion
		/// </summary>
		STOP_CAM = 0,
		/// <summary>
		/// Start cam motion
		/// </summary>
		START_CAM
	};

	/// <summary>
	/// Group Bond Option Enum
	/// </summary>
	enum class GROUPBONDOPTIONS : PMC_USINT
	{
		/// <summary>
		/// 6D bonded, but xy is decoupled from RZ
		/// </summary>
		SIXDOF_DECOUPLED = 0,
		/// <summary>
		/// 6D bonded, but xy is coupled with RZ
		/// </summary>
		SIXDOF_COUPLED,
		/*
		/// <summary>
		/// x,y,z,rz bonded, xy not coupled with rz
		/// </summary>
		XYZRZ_DECOUPLED,
		/// <summary>
		/// x,y,z,rz bonded, xy coupled with rz
		/// </summary>
		XYZRZ_COUPLED
		*/
	};

	/// <summary>
	/// short axes center mode Enum
	/// </summary>
	enum class SHORTAXESCENTERMODE : PMC_USINT
	{
		/// <summary>
		/// XBOT center is used for rotation or tilt calculations
		/// </summary>
		XBOT_CENTER = 0,
		/// <summary>
		/// User defines the center (x,y coordinates) that the XBOT rotates around
		/// </summary>
		USER_DEFINED_CENTER_XY,
	};

	/// <summary>
	/// Zone state
	/// </summary>
	enum class ZONESTATE : PMC_USINT
	{
		/// <summary>
		/// The zone has not been defined and cannot be used
		/// </summary>
		UNDEFINED = 0,
		/// <summary>
		/// The zone is activated, XBots can move freely inside the zone and may move across the zone boundary
		/// </summary>
		ACTIVATED_UNFENCED = 1,
		/// <summary>
		/// The zone is deactivating. Any XBots inside the zone are in the process of being deactivated
		/// </summary>
		DEACTIVATING = 2,
		/// <summary>
		/// The zone is a loading / unloading zone, a user may add or remove XBots from the zone manually. The XBots inside the zone are deactivated. XBots may not move across the zone boundary
		/// </summary>
		LOADING_FENCED = 3,
		/// <summary>
		/// The zone is activating. Any XBots inside the zone are in the process of being discovered / levitated
		/// </summary>
		ACTIVATING = 4,
		/// <summary>
		/// The zone is activated, XBots can move freely inside the zone, but may not move across the zone boundary
		/// </summary>
		ACTIVATED_FENCED = 5,
		/// <summary>
		/// The zone is in the process of being fenced. The system is attempting to create a fence that will stop XBots from crossing the zone boundary
		/// </summary>
		BUILDING_FENCE = 6,
	};

	/// <summary>
	/// zone operation performed by the Zone Control command. 0 = Deactivate Zone, 1= Activate Zone
	/// </summary>
	enum class ZONEOPERATION : PMC_USINT
	{
		/// <summary>
		/// Deactivate the zone, if successful, the zone is converted into a loading/unloading zone. Any XBots inside the zone will become deactivated
		/// </summary>
		DEACTIVATE_ZONE = 0,
		/// <summary>
		/// Activated the zone, if successful, the zone is converted into a fenced active zone. Any XBots inside the zone will be levitated
		/// </summary>
		ACTIVATE_ZONE = 1,
	};

	/// <summary>
	/// fence operation performed by the Zone Fence Control command. 0 = remove fence around the zone, 1= build fence around the zone
	/// </summary>
	enum class FENCEOPERATION : PMC_USINT
	{
		/// <summary>
		/// remove the fence around the zone, XBots can freely move across the zone boundary
		/// </summary>
		REMOVE_FENCE = 0,
		/// <summary>
		/// build the fence around the zone, XBots cannot move across the zone boundary
		/// </summary>
		BUILD_FENCE = 1,
	};

	/// <summary>
	/// queue operation performed by the Queue Control Command. 0=create queue, 1=delete queue
	/// </summary>
	enum class QUEUEOPERATION : PMC_USINT
	{
		/// <summary>
		/// Create a queue, queue definitions are required
		/// </summary>
		CREATE_QUEUE = 0,
		/// <summary>
		/// Delete the queueing area
		/// </summary>
		DELETE_QUEUE = 1,
	};

	/// <summary>
	/// select an area of the region to use
	/// </summary>
	enum class AREASELECTION : PMC_USINT
	{
		/// <summary>
		/// An area located at the bottom left corner of the queue, with the same size as the largest allowable XBot inside the area
		/// </summary>
		BOTTOM_LEFT = 0,
		/// <summary>
		/// An area located at the top left corner of the queue, with the same size as the largest allowable XBot inside the area
		/// </summary>
		TOP_LEFT = 1,
		/// <summary>
		/// An area located at the top right corner of the queue, with the same size as the largest allowable XBot inside the area
		/// </summary>
		TOP_RIGHT = 2,
		/// <summary>
		/// An area located at the bottom right corner of the queue, with the same size as the largest allowable XBot inside the area
		/// </summary>
		BOTTOM_RIGHT = 3,
	};

	/// <summary>
	/// operation performed by the auto loading zone control command
	/// </summary>
	enum class ALZONEOPERATION : PMC_USINT
	{
		/// <summary>
		/// Create a autoload zone
		/// </summary>
		CREATE_AUTOLOAD_ZONE = 0,
		/// <summary>
		/// Delete the auto load zone
		/// </summary>
		DELETE_AUTOLOAD_ZONE = 1,
		/// <summary>
		/// Activate the auto load zone
		/// </summary>
		ACTIVATE_AUTOLOAD_ZONE = 2,
		/// <summary>
		/// Deactivate the auto load zone
		/// </summary>
		DEACTIVATE_AUTOLOAD_ZONE = 3
	};

	/// <summary>
	/// define the auto loading zone as either loading or unloading
	/// </summary>
	enum class ALZONETYPE : PMC_USINT
	{
		/// <summary>
		/// unload from flyway zone, send XBot to external device
		/// </summary>
		UNLOAD_ZONE = 0,
		/// <summary>
		/// load to flyway zone, receive XBot from external device
		/// </summary>
		LOAD_ZONE = 1,
	};

	/// <summary>
	/// configured auto loading zone state
	/// </summary>
	enum class ALZONESTATE : PMC_USINT
	{
		/// <summary>
		/// auto loading zone has not been defined
		/// </summary>
		UNDEFINED = 0,
		/// <summary>
		/// auto loading zone has been defined
		/// </summary>
		DEFINED = 1,
		/// <summary>
		/// defined as auto unloading zone (removing XBot from flyway)
		/// </summary>
		UNLOADING_ZONE = 2,
		/// <summary>
		/// defined as auto loading zone (adding XBot to flyway)
		/// </summary>
		LOADING_ZONE = 3,
	};

	/// <summary>
	/// unloading behaviour, either non-stop unloading, or stop before unloading
	/// </summary>
	enum class ALZONEUNLOADMODE : PMC_USINT
	{
		/// <summary>
		/// XBot will come to a stop at the boundary before the unload operation
		/// </summary>
		STOP_BEFORE_UNLOAD = 0,
		/// <summary>
		/// XBot will not stop but smoothly transition its motion into an unload operation
		/// </summary>
		NONSTOP_UNLOAD
	};
	
	/// <summary>
	/// what feedback type has been configured
	/// </summary>
	enum class FEEDBACKTYPECONFIGURED : PMC_USINT
	{		
		/// <summary>
		/// no feedback has been configured
		/// </summary>
		NO_FEEDBACK = 0,		
		/// <summary>
		/// standard feedback has been configured
		/// </summary>
		STANDARD_FEEDBACK,		
		/// <summary>
		/// extended feedback has been configured
		/// </summary>
		EXTENDED_FEEDBACK
	};
}